/** @preserve Eloquence WEBDLG v2 Plugin Example: Datepicker
 (C) Copyright Marxmeier Software AG, 2021-2024
 $Id: webdlg-datepicker.js,v 29.6 2025/12/15 11:15:49 rhg Exp $
*/
eq.plugin("Datepicker", {
   // Attribute descriptor.
   atr : [
      [ "format", eq.atrGET|eq.atrSET|eq.atrSTR|eq.atrPRI ],
      [ "date",   eq.atrGET|eq.atrSET|eq.atrSTR|eq.atrPRI ]
   ],

   // Class properties.
   prp : {
      minVersion  : 1,
      classList   : 'datepicker',
      changeEvent : 'input pointerdown mousedown blur',
      actionEvent : 'blur',
      border      : true
   },

   // API: Constructor.
   apiInit : function(dlg, el) {
      var
         cls = this.constructor,
         firstDayOfWeek = cls.firstDayOfWeek,
         cal, ct, cn, names;

      if (cls.dayNames === undefined) {
         // First instance,
         // initialize input element format from application locale.
         var
            locale = eq.app.locale.tag,
            format, r;
         cls.formatInput = format = new Intl.DateTimeFormat(locale, {
            year : 'numeric',
            month : '2-digit',
            day : '2-digit',
            weekday : 'short'
         }).format;

         // Use 2000-12-31 to obtain input element scan order.
         if (r = cls.reScan.exec(format(new Date(2000, 11, 31)))) {
            cls.scanInputOrder = [];
            for (var i = 1, l = r.length; i < l; i++)
               switch (r[i]) {
                  case '12':
                     cls.scanInputOrder.push('M');
                     break;
                  case '31':
                     cls.scanInputOrder.push('d');
                     break;
                  default:
                     cls.scanInputOrder.push('y');
               }
         }

         // Setup day and month names from application locale.
         var
            day = 1,
            month = 0,
            date;
         // Year 2017 begins on Sunday.
         date = new Date(2017, month, day);

         // Day names.
         names = cls.dayNames = [];
         format = new Intl.DateTimeFormat(locale, {
            weekday : 'short'
         }).format;
         do {
            date.setDate(day);
            names.push(format(date));
         } while (++day <= 7);

         // Month names.
         names = cls.monthNames = [];
         format = new Intl.DateTimeFormat(locale, {
            month : 'long'
         }).format;
         do {
            date.setMonth(month);
            names.push(format(date));
         } while (++month <= 11);

         // Default strings.
         eq.strings.default.datepicker = {
            // 'Today' bottom line.
            today : {
               format : { dateStyle : 'long' },
               pre    : "Today is ",
               post   : ""
            }
         };
      }

      // Text input element.
      el.appendChild(ct = document.createElement('INPUT'));
      ct.type = 'text';
      ct.setAttribute('autocomplete', 'off');
      ct.setAttribute('autocorrect', 'off');
      ct.setAttribute('spellcheck', false);

      // Open/close button.
      el.appendChild(ct = document.createElement('DIV'));
      ct.className = 'btn';

      // Calendar box.
      el.appendChild(cal = document.createElement('DIV'));
      cal.className = 'cal';

      // Calendar caption.
      cal.appendChild(cn = document.createElement('DIV'));
      cn.className = 'caption';
      // Year backward button.
      cn.appendChild(ct = document.createElement('DIV'));
      ct.className = 'yb';
      // Month backward button.
      cn.appendChild(ct = document.createElement('DIV'));
      ct.className = 'mb';
      // Title.
      cn.appendChild(ct = document.createElement('DIV'));
      ct.className = 'title';
      // Month ahead button.
      cn.appendChild(ct = document.createElement('DIV'));
      ct.className = 'ma';
      // Year ahead button.
      cn.appendChild(ct = document.createElement('DIV'));
      ct.className = 'ya';

      // Calendar days.
      cal.appendChild(cn = document.createElement('DIV'));
      cn.className = 'days';

      // 'Today' bottom line.
      cal.appendChild(cn = document.createElement('DIV'));
      cn.className = 'btm';

      // Set default format.
      this.setFormat(null);
      // Set default date.
      cls.setDate(el, new Date());
   },

   // API: Set attribute value.
   apiSetAttr : function(dlg, id, el, atr, idx, val) {
      switch (atr) {
         case 'font':
            eq.c.DlgElement.setFont(eq.Dom.inputElement(el), val.font);
            break;

         case 'format':
            this.setFormat(val.s);
            break;

         case 'date': {
            var date = val.s ? this.scanDate(val.s) : new Date();
            this.selectedDate = date || undefined;
            this.constructor.setDate(el, date);
            break;
         }
      }
   },

   // API: Set element-specific 'sensitive' properties.
   apiSetSensitive : function(dlg, id, el, sensitive) {
      eq.Dom.inputElement(el).disabled = !sensitive;
   },

   // API: Filter and/or handle key event.
   apiOnKey : function(dlg, id, el, e) {
      var cal = el.querySelector('.cal');
      switch (e.key) {
         case 'Enter':
            if (!cal) {
               eq.app.closeOverlay(el, this);
               return true; // Consume event.
            }
            return false; // Pass event to default handler.
         case 'Escape':
         case 'Esc': // IE
            if (!cal)
               eq.app.closeOverlay(el);
            return true; // Consume event.

         // Navigation?
         case 'PageUp':
         case 'PageDown':
         case 'ArrowUp':
         case 'Up':
         case 'ArrowDown':
         case 'Down':
            if (cal) {
               this.prepareOpen(el, cal);
               this.constructor.open(dlg, el);
               return true; // Consume event.
            }
            break;
         case 'ArrowLeft':
         case 'Left':
         case 'ArrowRight':
         case 'Right':
            if (cal)
               return null; // Delegate event to target.
            break;
         default:
            // No.
            return false; // Pass event to default handler.
      }
      cal = document.body.querySelector('body>.eq-plugin.datepicker.cal');
      if (cal) {
         var sel = cal.querySelector('.cal>.days .selected');
         switch (e.key) {
            case 'PageUp': {
               var date = this.showDate;
               if (date !== undefined) {
                  date.setMonth(date.getMonth() - 1);
                  this.update(cal);
               }
               return true; // Consume event.
            }
            case 'PageDown': {
               var date = this.showDate;
               if (date !== undefined) {
                  date.setMonth(date.getMonth() + 1);
                  this.update(cal);
               }
               return true; // Consume event.
            }
            case 'ArrowUp':
            case 'Up':
               if (sel) {
                  var cl = (sel = sel.previousElementSibling).classList;
                  if (!cl.contains('label') && !cl.contains('empty'))
                     this.constructor.select(cal, sel);
                  return true; // Consume event.
               }
               break;
            case 'ArrowDown':
            case 'Down':
               if (sel) {
                  if (sel = sel.nextElementSibling)
                     this.constructor.select(cal, sel);
                  return true; // Consume event.
               }
               break;
            case 'ArrowLeft':
            case 'Left':
               if (sel) {
                  var prev = sel.parentNode.previousElementSibling;
                  if (prev && (sel = prev.children[eq.Dom.parentIndex(sel)])) {
                     var cl = sel.classList;
                     if (!cl.contains('label') && !cl.contains('empty'))
                        this.constructor.select(cal, sel);
                  }
                  return true; // Consume event.
               }
               break;
            case 'ArrowRight':
            case 'Right':
               if (sel) {
                  var next = sel.parentNode.nextElementSibling;
                  if (next && (sel = next.children[eq.Dom.parentIndex(sel)])) {
                     var cl = sel.classList;
                     if (!cl.contains('label') && !cl.contains('empty'))
                        this.constructor.select(cal, sel);
                  }
                  return true; // Consume event.
               }
         }
         var days = cal.querySelector('.cal>.days').children;
         for (var i = 0, l = days.length; i < l; i++)
            if (sel = days[i].children[1])
               if (!sel.classList.contains('empty')) {
                  this.constructor.select(cal, sel);
                  break;
               }
         return true; // Consume event.
      }
      return false; // Pass event to default handler.
   },

   // API: Process type-ahead key stroke.
   apiOnTypeAhead : function(dlg, id, el, key) {
      if (key.length === 1) {
         var
            ct = eq.Dom.inputElement(el),
            v = ct.value,
            s0 = ct.selectionStart,
            s1 = ct.selectionEnd;
         if (s0 > 0) {
            if (s1 < v.length)
               v = v.substring(0, s0) + key + v.substring(s1);
            else
               v = v.substring(0, s0) + key;
         }
         else if (s1 < v.length)
            v = key + v.substring(s1);
         else
            v = key;
         ct.value = v;
         return true;
      }
      return false;
   },

   // API: Change event occurred.
   apiOnChange : function(dlg, id, el, e) {
      if (e.type === 'pointerdown' || e.type === 'mousedown') {
         var
            cls = this.constructor,
            app = eq.app,
            dm = eq.Dom,
            t = e.target,
            cl = t.classList,
            cal, date;
         if (t && cl.contains('btn')) {
            if ((cl = el.classList).contains('eq-disabled'))
               return false;
            cal = el.querySelector('.cal');
            if (cl.contains('eq-focus') && app.setFocus(id, el) && cal) {
               // Focus set, event submitted,
               // open when Dialog becomes interactive again.
               this.prepareOpen(el, cal);
               app.pendingFocus = {
                  id : id,
                  act : cls.onPendingFocus
               };
               return false;
            }
            var m = { self: this, el : el, a : t };
            t.classList.add('click');
            if (cal) {
               this.prepareOpen(el, m.cal = cal);
               cls.open(dlg, el);
            }
            else
               m.close = true;
            app.moveCapture(m, cls.move, cls.moveDone);
            dm.consumeEvent(e);
            return false;
         }
         if (el.classList.contains('open')) {
            if (cal = cls.calendar(t)) {
               if (cl.contains('day')) {
                  cls.select(cal, t);
                  app.moveCapture({
                     self: this, el : el, cal : cal, close: true
                  }, cls.move, cls.moveDone);
               }
               else if (cl.contains('btm')) {
                  app.closeOverlay(el);
                  this.selectedDate = (date = new Date());
                  this.stringAttrChanged(el.id, 'input', 'date', null,
                     this.formatSelectedDate());
                  cls.setDate(el, date);
                  if (el.classList.contains('eq-rule'))
                     this.submit(el.id, el, 'change');
               }
               else if ((date = this.showDate) !== undefined) {
                  if (cl.contains('yb'))
                     date.setFullYear(date.getFullYear() - 1);
                  else if (cl.contains('mb'))
                     date.setMonth(date.getMonth() - 1);
                  else if (cl.contains('ma'))
                     date.setMonth(date.getMonth() + 1);
                  else if (cl.contains('ya'))
                     date.setFullYear(date.getFullYear() + 1);
                  else
                     date = null;
                  if (date !== null)
                     this.update(cal);
               }
            }
            while (t) {
               if (cl = t.classList) {
                  if (cl.contains('eq-overlay')) {
                     dm.consumeEvent(e);
                     return false;
                  }
                  if (cl.contains('eq-plugin')) {
                     app.closeOverlay(el);
                     return false;
                  }
               }
               t = t.parentNode;
            }
         }
      }
      else if (e.type === 'blur') {
         var
            cls = this.constructor,
            t = e.target,
            date;
         if (t.tagName === 'INPUT') {
            if (!el.querySelector('.cal'))
               app.closeOverlay(el, this);
            if (!cls.equalDate(date = cls.scanInput(t.value),
                               this.selectedDate)) {
               this.selectedDate = date ? date : undefined;
               this.stringAttrChanged(id, 'input', 'date', null,
                  this.formatSelectedDate());
               cls.setDate(el, date);
            }
         }
      }
      return true;
   },

   // API: Obtain focus element.
   apiFocusElement : function(dlg, el) {
      return eq.Dom.inputElement(el);
   },

   // API: Plugin has gained focus.
   apiFocusGained : function(dlg, ct) {
      // Select on focus except on mouse click or if focus doesn't change.
      if (   eq.app.pendingMouseDown === undefined
          && ct !== document.activeElement) {
         ct.select();
         ct.scrollLeft = ct.scrollWidth;
      }
   },

   // API: Plugin has lost focus.
   apiFocusLost : function(dlg, el) {
      if (el.classList.contains('open'))
         eq.app.closeOverlay(el);
      var ct = eq.Dom.inputElement(el);
      ct.selectionStart = ct.selectionEnd = 0;
      ct.blur();
   },

   // Implementation.
   // ---------------

   // Prepare to open calendar.
   prepareOpen : function(el, cal) {
      var
         cls = this.constructor,
         date = cls.scanInput(eq.Dom.inputElement(el).value);
      if (!cls.equalDate(date, this.selectedDate)) {
         this.selectedDate = date ? date : undefined;
         this.stringAttrChanged(el.id, 'input', 'date', null,
            this.formatSelectedDate());
         cls.setDate(el, date);
      }
      this.showDate = undefined;
      this.update(cal);
   },

   // Set date format.
   setFormat : function(f) {
      if (!f || f.length === 0)
         f = "dd.MM.yyyy";
      invalid: {
         var r = this.constructor.reSetFormat.exec(f);
         if (r) {
            var o = [], y, m, d;
            for (var i = 2; i <= 6; i += 2)
               switch (r[i].charAt(0)) {
                  case 'y':
                     if (y !== undefined)
                        break invalid;
                     y = i;
                     o.push('y');
                     break;
                  case 'M':
                     if (m !== undefined)
                        break invalid;
                     m = i;
                     o.push('M');
                     break;
                  case 'd':
                     if (d !== undefined)
                        break invalid;
                     d = i;
                     o.push('d');
                     break;
                  default:
                     break invalid;
               }
            this.format = {
               r : r,
               y : y,
               m : m,
               d : d,
               o : o
            };
         }
      }
   },

   // Format selected date.
   formatSelectedDate : function() {
      var
         date = this.selectedDate,
         format = this.format,
         r, s, i;
      if (!date || !format)
         return '';

      r = format.r;
      s = r.slice(1);

      var y = date.getFullYear();
      if (r[i = format.y].length < 4)
         y %= 100;
      s[i-1] = y;

      var m = date.getMonth() + 1;
      if (r[i = format.m].length > 1)
         m = ('00' + m).slice(-2);
      s[i-1] = m;

      var d = date.getDate();
      if (r[i = format.d].length > 1)
         d = ('00' + d).slice(-2);
      s[i-1] = d;

      return s.join('');
   },

   // Scan date according to format.
   scanDate : function(s) {
      var
         cls = this.constructor,
         format = this.format,
         r = cls.reScan.exec(s);
      if (r && format) {
         var
            scanOrder = format.o,
            y, m, d;
         for (var i = 1, l = r.length; i < l; i++)
            switch (scanOrder[i-1]) {
               case 'M':
                  m = Number(r[i])-1;
                  break;
               case 'd':
                  d = Number(r[i]);
                  break;
               default:
                  if ((y = Number(r[i])) < 100) {
                     if (y <= cls.twoDigitYearThreshold)
                        y += cls.currentCentury;
                     else
                        y += cls.currentCentury - 100;
                  }
            }
         return new Date(y, m, d);
      }
      return null;
   },

   // Update calendar.
   update : function(cal) {
      var selectedDate = this.selectedDate || new Date();
      if (this.showDate === undefined)
         this.showDate = new Date(selectedDate);

      var
         cls = this.constructor,
         showDate = this.showDate,
         year = showDate.getFullYear(),
         month = showDate.getMonth();

      // Title.
      cal.querySelector('.cal>.caption>.title').textContent =
         cls.monthNames[month] + ' ' + year;

      // Calendar days.
      var
         firstDayOfWeek = cls.firstDayOfWeek,
         names = cls.dayNames,
         days = cal.querySelector('.cal>.days'),
         temp = new Date(showDate),
         now = new Date(),
         selectedDay, today, col, cols, day, cn, ct, cl;

      if (   selectedDate.getFullYear() === year
          && selectedDate.getMonth() === month)
         selectedDay = selectedDate.getDate();

      if (now.getFullYear() === year && now.getMonth() === month)
         today = now.getDate();

      // (Re-)build calendar.
      days.textContent = '';
      for (col = 0; col < 7; col++) {
         days.appendChild(cn = document.createElement('DIV'));
         cn.appendChild(ct = document.createElement('DIV'));
         ct.className = 'label';
         ct.textContent = names[(col + firstDayOfWeek) % 7];
      }
      cols = days.children;
      for (col = 0, day = 1;; day++) {
         temp.setDate(day);
         if (temp.getMonth() !== month)
            break;

         if (day === 1) {
            // First day in month, fill preceding empty days.
            var emptyDays = (temp.getDay() + 7 - firstDayOfWeek) % 7;
            while (emptyDays--) {
               cols[col].appendChild(ct = document.createElement('DIV'));
               ct.className = 'day empty';
               if (++col === 7)
                  col = 0;
            }
         }

         cols[col].appendChild(ct = document.createElement('DIV'));
         (cl = ct.classList).add('day');
         if (day === today)
            cl.add('today');
         if (day === selectedDay)
            cl.add('selected');
         ct.textContent = day;
         if (++col === 7)
            col = 0;
      }

      // 'Today' bottom line.
      cal.querySelector('.cal>.btm').textContent
         = eq.strings.get('datepicker.today.pre')
         + new Intl.DateTimeFormat(eq.app.locale.tag,
                  eq.strings.get('datepicker.today.format')).format(now)
         + eq.strings.get('datepicker.today.post');
   },

   // Static methods.
   // ---------------
   static : {

      // Static API: Class installation.
      install : function() {
         this.firstDayOfWeek = 1; // 0=Sunday, 1=Monday.
         var
            now = new Date(),
            y = now.getFullYear();
         this.currentCentury = Math.floor(y / 100) * 100;
         this.twoDigitYearThreshold = (y - 80) % 100;
      },

      // Static API: Dispose static class properties.
      dispose : function() {
         this.formatInput = undefined;
         this.scanInputOrder = undefined;
         this.dayNames = undefined;
         this.monthNames = undefined;
      },

      // Static: Open calendar.
      open : function(dlg, el) {
         var cl = el.classList;
         if (!cl.contains('open')) {
            var
               dm = eq.Dom,
               ct = dm.inputElement(el),
               rc = el.getBoundingClientRect(),
               cal = el.querySelector('.cal'),
               s;

            // Set overlay position.
            (s = cal.style).left = (rc.left + window.pageXOffset) + 'px';
            s.top = (rc.top + rc.height + window.pageYOffset - 1) + 'px';

            // Set colors and font.
            dm.copyColors(cal, el);
            if (ct.classList.contains('eq-font'))
               dm.copyFont(cal, ct);
            else
               dlg.copyFontToElement(cal);

            // Make overlay.
            cal.className = 'eq-plugin datepicker cal';
            if (cl.contains('eq-border'))
               cal.classList.add('eq-border');
            eq.app.makeOverlay(el, cal, function(el, cal, self) {
               // Overlay closed.
               var cl = el.classList, date, sel;
               if (!cl.contains('open'))
                  throw new Error("Datepicker.close BUG: not open, id:" + el.id);
               el.appendChild(cal);
               cal.className = 'cal';
               cl.remove('open');
               if (self !== undefined) {
                  if (   (date = self.showDate) !== undefined
                      && (sel = cal.querySelector('.cal>.days .selected'))) {
                     // Calendar selection committed.
                     date.setDate(sel.textContent);
                     self.selectedDate = date;
                     self.stringAttrChanged(el.id, 'input', 'date', null,
                        self.formatSelectedDate());
                     self.constructor.setDate(el, date);
                     if (el.classList.contains('eq-rule'))
                        self.submit(el.id, el, 'change');
                  }
                  self.showDate = undefined;
               }
            });

            cl.add('open');
         }
      },

      // Static: Pending focus, open calendar when Dialog becomes interactive.
      onPendingFocus : function(el) {
         var
            dlg = eq.app.currDlg,
            self = dlg.eds.get(el.id);
         self.constructor.open(dlg, el);
      },

      // Static: Obtain calendar container element.
      calendar : function(el) {
         var cl;
         while (el) {
            if ((cl = el.classList) && cl.contains('cal'))
               return el;
            el = el.parentNode;
         }
         return null;
      },

      // Static: Scan date from input element.
      scanInput : function(s) {
         var r = this.reScan.exec(s);
         if (r) {
            var
               scanOrder = this.scanInputOrder,
               y, m, d;
            for (var i = 1, l = r.length; i < l; i++)
               switch (scanOrder[i-1]) {
                  case 'M':
                     m = Number(r[i])-1;
                     break;
                  case 'd':
                     d = Number(r[i]);
                     break;
                  default:
                     if ((y = Number(r[i])) < 100) {
                        if (y <= this.twoDigitYearThreshold)
                           y += this.currentCentury;
                        else
                           y += this.currentCentury - 100;
                     }
               }
            return new Date(y, m, d);
         }
         return null;
      },

      // Static: Set date to input element.
      setDate : function(el, date) {
         eq.Dom.inputElement(el).value = date ? this.formatInput(date) : '';
      },

      // Static: Check if dates are equal.
      equalDate : function(d1, d2) {
         if (!d1 || !d2)
            return false;
         if (d1.getFullYear() !== d2.getFullYear())
            return false;
         if (d1.getMonth() !== d2.getMonth())
            return false;
         if (d1.getDate() !== d2.getDate())
            return false;
         return true;
      },

      // Static: Select calendar day.
      select : function(cal, day) {
         // Deselect.
         var sel = cal.querySelectorAll('.cal>.days .selected');
         for (var i = 0, l = sel.length; i < l; i++)
            sel[i].classList.remove('selected');
         // Select
         if (day !== undefined)
            day.classList.add('selected');
      },

      // Static: Handle Datepicker move capture.
      move : function(e, m) {
         var cal = m.cal, t;
         if (cal && (t = e.target).classList.contains('day')) {
            m.close = true;
            m.self.constructor.select(cal, t);
         }
      },
      moveDone : function(m) {
         if (m.close)
            eq.app.closeOverlay(m.el, m.self);
         var a = m.a;
         if (a !== undefined)
            a.classList.remove('click');
         return true;
      },

      // setFormat() expression.
      reSetFormat : new RegExp(
         "^(.*?)(d+|M+|y+)(.*?)(d+|M+|y+)(.*?)(d+|M+|y+)(.*?)$"),

      // Common scan expression.
      reScan : new RegExp("([0-9]+).*?([0-9]+).*?([0-9]+)")
   }
});
